#!/bin/sh
#---------------------------------------------------------
#
# RR-1 MQI Local Bindings(Thread)
#
# Parameters:
# $1 : Queue manager name
# $2 : Number of connections 
# $3 : Messages
# $4 : Message Bytes
# $5 : Log Directory 
# $6 : Workload
# $7 : MQ Constant
# $8 : Data File
#
# Environments(./runpfmts.ini):
# $INCL_CON_TIME : Include MQCONN/MQDISC time in measurement,
#                  0:not include, 1:include
# $RCV_BUFF_SIZE : Receive buffer size
# $CON_RETRY     : The number of connection retry
# $CON_INTERVAL  : Connection retry interval
# $RR1_RequestQ  : Request queue name
# $RR1_ReplyQ    : Reply queue name
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#
#---------------------------------------------------------
# return code
RET_OK=0
RET_NG=1

USAGE="$0 <Qmgr> <Connections> <Messages> <Bytes> <Log Directory> <Workload> <MQ Constant> <Data File>"

if (( $# < 6 ))
then
  printf "\nUsage: $USAGE\n\n"

  cat <<-!
    RR-1 MQI Local Bindings(Thread)

    Parameters:
    \$1 : Queue manager name
    \$2 : Number of connections 
    \$3 : Number of messages
    \$4 : Message Bytes
    \$5 : Log Directory 
    \$6 : Workload
    \$7 : MQ Constant (ex. MQPER_PERSISTENT, MQPER_NOT_PERSISTENT)
    \$8 : Data File (Input message file) (Option)
         If not specified, "<workload name>" is used as the message.

    Environments(./runpfmts.ini):
    \$INCL_CON_TIME : Include MQCONN/MQDISC time in measurement,
                     0:not include, 1:include
    \$RCV_BUFF_SIZE : Receive buffer size
    \$CON_RETRY     : The number of connection retry
    \$CON_INTERVAL  : Connection retry interval
    \$RR1_RequestQ  : Request queue name
    \$RR1_ReplyQ    : Reply queue name

!

  exit $RET_NG
fi

# source environments for performance test
. ./runpfmts.ini

# import library
. ./lib/libpfmts.sh

TARGET_QMGR=$1
Connections=$2
Messages=$3
Bytes=$4
LogDirPre=$5
Workload=$6
MQ_Constant=$7
DataFile=$8

TRM=""
if [[ $INCL_CON_TIME -eq "1" ]]
then
    TRM="-tr"
fi

LogDir=$LogDirPre/${Workload}_${Bytes}_${Connections}_${TARGET_QMGR}_`date "+%Y%m%d_%H%M%S"`

mkdir -p $LogDir

RequestLogNamePre=$LogDir/req
ReplyLogNamePre=$LogDir/rep

logNameReq=${RequestLogNamePre}_`date "+%Y%m%d_%H%M%S"`.log
logNameRep=${ReplyLogNamePre}_`date "+%Y%m%d_%H%M%S"`.log

((Total_Messages = Connections * Messages))

echo ""
echo "Start Responder Program, Get $RR1_RequestQ, Put $RR1_ReplyQ"
# Responder
# primary side: MQGMO_ACCEPT_TRUNCATED_MSG
# secondary side: -l $Bytes

ArgumentListRep="-qm $TARGET_QMGR -q $RR1_RequestQ -oq $RR1_ReplyQ MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -n $Messages -cr $CON_RETRY -wp $CON_INTERVAL -ni $Connections -sz $RCV_BUFF_SIZE $TRM -ss -l $Bytes MQMT_REPLY $MQ_Constant MQPMO_NO_SYNCPOINT -im"

echo "mqpgf $ArgumentListRep"
nohup mqpgf $ArgumentListRep 2>&1 | tee $logNameRep > /dev/null &

echo ""
echo "Start Requester Program, Put $RR1_RequestQ, Get $RR1_ReplyQ"
# Requester
# secondary side: MQGMO_ACCEPT_TRUNCATED_MSG
if [[ -z $DataFile ]]
then
  ArgumentListReq="-qm $TARGET_QMGR -q $RR1_RequestQ -m \"$Workload\" -iq $RR1_ReplyQ -rq $RR1_ReplyQ MQMT_REQUEST $MQ_Constant MQPMO_NO_SYNCPOINT -n $Messages -l $Bytes -cr $CON_RETRY -wp $CON_INTERVAL -ni $Connections $TRM -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
else
  ArgumentListReq="-qm $TARGET_QMGR -q $RR1_RequestQ -f $DataFile -iq $RR1_ReplyQ -rq $RR1_ReplyQ MQMT_REQUEST $MQ_Constant MQPMO_NO_SYNCPOINT -n $Messages -l $Bytes -cr $CON_RETRY -wp $CON_INTERVAL -ni $Connections $TRM -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
fi

echo "mqpgf $ArgumentListReq"
nohup mqpgf $ArgumentListReq 2>&1 | tee $logNameReq > /dev/null &

flag=0
while :
do
  # There is no need to check 'tee' process on the secondary side.
  ps -ef|grep -E "mqpgf.*$TARGET_QMGR.*$RR1_RequestQ.*$RR1_ReplyQ|tee.*$logNameReq"|grep -v grep > /dev/null

  if (( $? == 1 ))
  then
    printf "terminated."
    break
  fi
  sleep 1
  if ((flag == 0))
  then
    printf "Waiting for the completion of the test process."
  else
    printf "."
  fi
  ((flag += 1))
done

echo ""

fnGetElapsedTime_Thread $logNameReq $Total_Messages | tee -a $LogDirPre/console.log

exit $RET_OK

