# Windows PwowerShell
#---------------------------------------------------------
#
# RR-5 MQI Distributed Queuing (for Requester)
#
# Parameters:
# $1 : Queue manager name
# $2 : Number of connections
# $3 : Messages
# $4 : Message Bytes
# $5 : Log Directory
# $6 : Workload
# $7 : MQ Constant
# $8 : Data File
#
# Environments(./runpfmts.ini):
# $INCL_CON_TIME      : Include MQCONN/MQDISC time in measurement,
#                       0:not include, 1:include
# $RCV_BUFF_SIZE      : Receive buffer size
# $CON_RETRY          : The number of connection retry
# $CON_INTERVAL       : Connection retry interval
# $RR5_RequestQPrefix : Request queue name prefix
# $RR5_ReplyQPrefix   : Reply queue name prefix
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#
#---------------------------------------------------------
param(
    [string]$TARGET_QMGR = "",
    [int]$Connections = 0,
    [int]$Messages = 0,
    [int]$Bytes = 0,
    [string]$LogDirPre = "",
    [string]$Workload = "",
    [string]$MQ_Constant = "",
    [string]$DataFile = ""
)

# return code
[int]$RET_OK=0
[int]$RET_NG=1

$USAGE = $myInvocation.MyCommand.name + " <Qmgr> <Connections> <Messages> <Bytes> <Log Directory> <Workload> <MQ Constant> <Data File>"

if ($PSBoundParameters.Count -lt 6)
{
    Write-output "" "Usage: $USAGE" "" ""

    $usage_document = @"
    RR-5 MQI Distributed Queuing (for Requester)

    Parameters:
    `$1 : Queue manager name
    `$2 : Number of connections
    `$3 : Number of messages
    `$4 : Message Bytes
    `$5 : Log Directory
    `$6 : Workload
    `$7 : MQ Constant (ex. MQPER_PERSISTENT, MQPER_NOT_PERSISTENT)
    `$8 : Data File (Input message file) (Option)
         If not specified, "<workload name>" is used as the message.

    Environments(./runpfmts.bat):
    `$INCL_CON_TIME      : Include MQCONN/MQDISC time in measurement,
                          0:not include, 1:include
    `$RCV_BUFF_SIZE      : Receive buffer size
    `$CON_RETRY          : The number of connection retry
    `$CON_INTERVAL       : Connection retry interval
    `$RR5_RequestQPrefix : Request queue name prefix
    `$RR5_ReplyQPrefix   : Reply queue name prefix

"@

    Write-output $usage_document

    exit $RET_NG
}

# import library
. ./lib/libpfmts.ps1

# source environments for performance test
fnSourceEnvironments

[string]$RR5_RequestQPrefix=${env:RR5_RequestQPrefix}
[string]$RR5_ReplyQPrefix=${env:RR5_ReplyQPrefix}
[int]$CON_RETRY=${env:CON_RETRY}
[int]$CON_INTERVAL=${env:CON_INTERVAL}
[int]$RCV_BUFF_SIZE=${env:RCV_BUFF_SIZE}
[int]$INCL_CON_TIME=${env:INCL_CON_TIME}

$TRM = ""
$SFILE = "syncstart"
$SRF = "-sf ${SFILE}"
if ($INCL_CON_TIME -eq 1)
{
    $TRM = "-tr"
    $SRF = ""
}

$dateTime = Get-Date -Format "yyyyMMdd_HHmmss"
$LogDir = "${LogDirPre}\${Workload}_${Bytes}_${Connections}_${TARGET_QMGR}_${dateTime}"

mkdir $LogDir

$LogNamePre = "${LogDir}\req"
$StderrLogNamePre = "${LogDir}\stderr_req"

Write-Host ""
Write-Host "TARGET_QMGR: $TARGET_QMGR"
Write-Host "Connections: $Connections Messages: $Messages Bytes: $Bytes LogDirPre: $LogDirPre Workload: $Workload MQ_Constant: $MQ_Constant DataFile: $DataFile"

$Total_Messages = $Connections * $Messages

$WaitProcess = @()

if (Test-Path $SFILE)
{
    Write-Output "delete $SFILE file for synchronization start."
    Write-Output ""
    del $SFILE
}

# Requester
# secondary side: MQGMO_ACCEPT_TRUNCATED_MSG
for ( $Counter = 1; $Counter -le $Connections; $Counter++ )
{
    $RequestQ = $RR5_RequestQPrefix + $Counter
    $ReplyQ = $RR5_ReplyQPrefix + $Counter

    if ($DataFile -eq "")
    {
        $ArgumentList = "-qm $TARGET_QMGR -q $RR5_RequestQPrefix$Counter -m $Workload -iq $RR5_ReplyQPrefix$Counter -rq $RR5_ReplyQPrefix$Counter MQMT_REQUEST $MQ_Constant MQPMO_NO_SYNCPOINT -n $Messages -l $Bytes $SRF -cr $CON_RETRY -wp $CON_INTERVAL $TRM -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
    }
    else
    {
        $ArgumentList = "-qm $TARGET_QMGR -q $RR5_RequestQPrefix$Counter -f $DataFile -iq $RR5_ReplyQPrefix$Counter -rq $RR5_ReplyQPrefix$Counter MQMT_REQUEST $MQ_Constant MQPMO_NO_SYNCPOINT -n $Messages -l $Bytes $SRF -cr $CON_RETRY -wp $CON_INTERVAL $TRM -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
    }

    $dateTime = Get-Date -Format "yyyyMMdd_HHmmss"
    $logName = "${LogNamePre}_${Counter}_${dateTime}.log"
    $stderrLogName = "${StderrLogNamePre}_${Counter}_${dateTime}.log"

    Write-Host "Start Requester Program, Put $RequestQ, Get $ReplyQ"
    Write-Host "mqpgf $ArgumentList"
    $WaitProcess += Start-Process -FilePath mqpgf.exe -RedirectStandardOutput $logName -RedirectStandardError $stderrLogName -PassThru -NoNewWindow -ArgumentList $ArgumentList
    Write-Host "start background: $Counter of $Connections."
}

if ($SRF -ne "")
{
    New-Item $SFILE -type file 
    Write-Output ""
}

$index = 0
for ( ; $index -lt $WaitProcess.count; $index++)
{
    do
    {
        try
        {
            $process = Get-Process -Id $WaitProcess[$index].Id -ErrorAction stop | Format-Table
            Write-Output "The number of the sampling process displayed: $index"
            $process
        }
        catch
        {
            Write-Output "The end of the test command has been detected. Process Number: $index"
            break
        }
        Start-Sleep 5
    }
    while(1)
}

Write-Host ""
fnGetElapsedTime -LOGFILE_PREFIX "$LogNamePre" -MESSAGES $Messages -CONNECTIONS $Connections | tee-object ${LogDirPre}/console.log -Append

exit $RET_OK
